/*
 * Written by Dawid Kurzyniec and released to the public domain, as explained
 * at http://creativecommons.org/licenses/publicdomain
 */

package edu.emory.mathcs.util.collections.longs;

import java.util.*;

/**
 * Sorted set is a collection with no duplicate elements. Primitive sets have
 * some features not found in object sets. A primitive set is
 * associated with a <em>domain</em> with boundaries defined by {@link #min()}
 * and {@link #max()}. All numbers contained within this set must fit
 * between min and max, inclusive. Attempt to add a number from outside the
 * domain will have no effect.
 *
 * <p>
 * It is possible to obtain a complement view of a
 * primitive set, using {@link #complementSet()}. The complement view contains
 * all numbers between min and max, inclusive, which are not contained in
 * this set. (In other words, complement view never contains numbers from
 * outside of the domain of this set).
 *
 * <p>
 * Contrary to standard Java collections, intervals in this primitive collection
 * package are <em>inclusive</em> on both sides. In other words, interval
 * [min, max] contains all numbers c such that min <= c <= max. (Otherwise,
 * MAX_VALUE could not be put in the set).
 *
 * <p>
 * It is possible to obtain a subset view of a sorted set, using one of
 * the methods {@link #subSet()}, {@link #headSet()}, and {@link #tailSet()}.
 * The subset view narrows the domain, and its size is thus never bigger than
 * that of this set.
 *
 * <p>
 * Sorted sets define a family of navigation methods, analogous to that
 * found in {@link edu.emory.mathcs.backport.java.util.NavigableSet}.
 *
 * @author Dawid Kurzyniec
 * @version 1.0
 */
public interface LongSortedSet extends LongSet {

    /**
     * A subset view containing all elements from this set between first,
     * inclusive, and last, inclusive. More precisely, the view is narrowed to
     * the domain [min, max]. Hence, complement set of this set will NOT
     * include any elements outside [min, max].
     *
     * @param first the minimum element of this view (inclusive).
     * @param last the maximum element of this view (inclusive).
     * @return the subset view
     */
    LongSortedSet subSet(long first, long last);

    /**
     * A subset view containing all elements from this set that are not greater
     * than last. More precisely, the view is narrowed to
     * the domain [MIN_VALUE, max]. Hence, complement set of this set will NOT
     * include any elements > max.
     *
     * @param last the maximum element of this view (inclusive).
     * @return the headset view
     */
    LongSortedSet headSet(long last);

    /**
     * A subset view containing all elements from this set that are not smaller
     * than first. More precisely, the view is narrowed to
     * the domain [min, MAX_VALUE]. Hence, complement set of this set will NOT
     * include any elements < min.
     *
     * @param first the minimum element of this view (inclusive).
     * @return the tailset view
     */
    LongSortedSet tailSet(long first);

    /**
     * Returns the smallest number in this set.
     *
     * @return the smallest number in this set.
     * @throws NoSuchElementException if the set is empty.
     */
    long first();

    /**
     * Returns the largest number in this set.
     *
     * @return the largest number in this set.
     * @throws NoSuchElementException if the set is empty.
     */
    long last();

    /**
     * Returns the smallest number in this set >= e.
     *
     * @return the smallest number in this set >= e.
     * @throws NoSuchElementException if all elements are < e.
     */
    long ceiling(long e);

    /**
     * Returns the smallest number in this set > e.
     *
     * @return the smallest number in this set > e.
     * @throws NoSuchElementException if all elements are <= e.
     */
    long higher(long e);

    /**
     * Returns the largest number in this set <= e.
     *
     * @return the largest number in this set <= e.
     * @throws NoSuchElementException if all elements are > e.
     */
    long floor(long e);

    /**
     * Returns the largest number in this set < e.
     *
     * @return the largest number in this set < e.
     * @throws NoSuchElementException if all elements are >= e.
     */
    long lower(long e);

    /**
     * Returns and removes the smallest number in this set.
     *
     * @return the smallest number in this set.
     * @throws NoSuchElementException if the set is empty.
     */
    long pollFirst();

    /**
     * Returns and removes the largest number in this set.
     *
     * @return the smallest number in this set.
     * @throws NoSuchElementException if the set is empty.
     */
    long pollLast();

    /**
     * Returns the minimum count of intervals into which this set can be
     * decomposed. For instance, {1, 3,4,5, 8,9}.intervalCount() == 3.
     *
     * @return int the interval count of this set.
     */
    int intervalCount();

    /**
     * Returns an iterator over intervals of this set, in an increasing
     * numerical order.
     *
     * @return an iterator over intervals of this set
     */
    Iterator intervalIterator();

    /**
     * Returns an iterator over intervals of this set, in a decreasing
     * numerical order.
     *
     * @return a descending iterator over intervals of this set
     */
    Iterator descendingIntervalIterator();

    /**
     * Returns an iterator over numbers in this set, in an increasing
     * numerical order.
     *
     * @return an iterator over numbers in this set
     */
    LongIterator iterator();

    /**
     * Returns an iterator over numbers in this set, in a decreasing
     * numerical order.
     *
     * @return a descending iterator over numbers in this set
     */
    LongIterator descendingIterator();

    /**
     * Returns the first (left-most), widest interval contained in this set,
     * or null if this set is empty.
     * @return the first interval contained in this set
     */
    LongInterval firstInterval();

    /**
     * Returns the last (right-most), widest interval contained in this set,
     * or null if this set is empty.
     * @return the last interval contained in this set
     */
    LongInterval lastInterval();

    /**
     * Returns the widest interval contained in this set that includes the
     * specified number, or null if this set does not include the specified
     * number.
     *
     * @return the interval containing the specified number.
     */
    LongInterval enclosingInterval(long e);

    /**
     * Returns the largest and widest interval contained in this set which
     * all elements are strictly less than the specified number.
     *
     * @return the largest interval smaller than the specified number.
     */
    LongInterval lowerInterval(long e);

    /**
     * Returns the smallest (left-most), widest interval contained in this set
     * which all elements are strictly greater than the specified number.
     *
     * @return the smallest interval greater than the specified number.
     */
    LongInterval higherInterval(long e);

    /**
     * Returns the largest (right-most), widest interval contained in this set
     * which elements are not all greater than the specified number. In other
     * words, it either includes the specified number or has all elements
     * strictly less than the specified number.
     *
     * @return the largest interval which lower bound is <= than the specified
     *         number.
     */
    LongInterval floorInterval(long e);

    /**
     * Returns the smallest (left-most), widest interval contained in this set
     * which elements are not all smaller than the specified number. In other
     * words, it either includes the specified number or has all elements
     * strictly greater than the specified number.
     *
     * @return the smallest interval which upper bound is >= than the specified
     *         number.
     */
    LongInterval ceilingInterval(long e);

    /**
     * Returns and removes the first (left-most), widest interval contained in
     * this set, or null if this set is empty.
     * @return the first interval contained in this set
     */
    LongInterval pollFirstInterval();

    /**
     * Returns and removes the last (right-most), widest interval contained in
     * this set, or null if this set is empty.
     * @return the last interval contained in this set
     */
    LongInterval pollLastInterval();

    String toString();

    /**
     * Returns the interval-string representation of this set (suitable for
     * representing sets that are possibly large but clustered).
     * @return String
     */
    String toCompactString();
}
